<?php
/**
 * PropertyReader.php 2020-4-2
 * Gambio GmbH
 * http://www.gambio.de
 * Copyright (c) 2020 Gambio GmbH
 * Released under the GNU General Public License (Version 2)
 * [http://www.gnu.org/licenses/gpl-2.0.html]
 */

namespace Gambio\Shop\Properties\Database\Readers;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DBALException;
use Gambio\Shop\Product\ValueObjects\ProductId;
use Gambio\Shop\Properties\Database\Readers\Interfaces\PropertyReaderInterface;
use Gambio\Shop\Properties\ProductModifiers\Database\ValueObjects\PropertyModifierIdentifier;
use Gambio\Shop\Properties\Properties\Collections\CombinationCollection;
use Gambio\Shop\Properties\Properties\Collections\CombinationCollectionInterface;
use Gambio\Shop\Properties\Properties\Entities\Combination;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationEan;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationId;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationModel;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationOrder;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationQuantity;
use Gambio\Shop\Properties\Properties\ValueObjects\CombinationSurcharge;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\SellingUnitId;
use Gambio\Shop\SellingUnit\Unit\ValueObjects\Vpe;

/**
 * Class PropertyReader
 * @package Gambio\Shop\Properties\Database\Readers
 */
class PropertyReader implements PropertyReaderInterface
{

    /**
     * @var array
     */
    protected $cache = [
        'getCombinationModifierIds' => [],
        'hasProperties' => []
    ];

    /**
     * @var Connection
     */
    private $connection;


    /**
     * PropertyReader constructor.
     *
     * @param Connection $connection
     */
    public function __construct(Connection $connection)
    {
        $this->connection = $connection;
    }


    /**
     * @inheritDoc
     */
    public function getCombinationModifierIds(int $combinationId): iterable
    {

        $result = $this->getCombinationModifierIdsCached($combinationId);
        if ($result === null) {
            $result = [];
            $imagesData = $this->connection->createQueryBuilder()
                ->select('distinct products_id, properties_values_id')
                ->from('products_properties_index')
                ->where('products_properties_index.products_properties_combis_id = :combination_id')
                ->setParameter('combination_id', $combinationId)
                ->execute()
                ->fetchAll();
            foreach ($imagesData as $row) {
                if (empty($result[(int)$row['products_id']])) {
                    $result[(int)$row['products_id']] = [];
                }
                $result[(int)$row['products_id']][] = (int)$row['properties_values_id'];
            }
            $this->setCombinationModifierIdsCached($combinationId, $result);
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function getCombinationsFor(SellingUnitId $id): CombinationCollectionInterface
    {


        $where = '';
        $having = '';
        $result = new CombinationCollection();
        $propertyIdList = [];
        foreach ($id->modifiers() as $modifierId) {
            if ($modifierId instanceof PropertyModifierIdentifier) {
                $propertyIdList[] = $modifierId->value();
            }
        }
        $listCount = count($propertyIdList);
        if ($listCount) {
            $inList = implode(',', $propertyIdList);
            $where = "\n WHERE ppcv.properties_values_id IN ($inList) \n";
            $having = "\n HAVING COUNT(ppcv.products_properties_combis_values_id) = $listCount \n";
        }
        $productId = $id->productId()->value();

        $sql = "
            SELECT `ppc`.`products_properties_combis_id`,
                   `ppc`.`products_id`,
                   `ppc`.`sort_order`,
                   `ppc`.`combi_model`,
                   `ppc`.`combi_ean`,
                   `ppc`.`combi_quantity`,
                   `ppc`.`combi_shipping_status_id`,
                   `ppc`.`combi_weight`,
                   `ppc`.`combi_price_type`,
                   `ppc`.`combi_price`,
                   `ppc`.`products_vpe_id`,
                   `ppc`.`vpe_value`,
                   `pvpe`.`products_vpe_name`
            FROM `products_properties_combis_values` AS `ppcv`
            INNER JOIN `products_properties_combis` AS `ppc`
             ON `ppc`.`products_properties_combis_id` = `ppcv`.`products_properties_combis_id`
             AND ppc.products_id = {$productId}
            LEFT JOIN `products_vpe` as `pvpe`
             ON `ppc`.`products_vpe_id` = `pvpe`.`products_vpe_id`
             AND `pvpe`.`language_id` = {$this->activeLanguageId()}
            {$where}
            GROUP BY
                `ppc`.`products_properties_combis_id`,
                   `ppc`.`products_id`,
                   `ppc`.`sort_order`,
                   `ppc`.`combi_model`,
                   `ppc`.`combi_ean`,
                   `ppc`.`combi_quantity`,
                   `ppc`.`combi_shipping_status_id`,
                   `ppc`.`combi_weight`,
                   `ppc`.`combi_price_type`,
                   `ppc`.`combi_price`,
                   `ppc`.`products_vpe_id`,
                   `ppc`.`vpe_value`
            {$having}
            ";
        $data = $this->connection->query($sql)->fetchAll();

        $nonLinearSurcharge = (count($data) > 0) ? $this->hasNonLinearSurcharge($productId) : false;

        foreach ($data as $row) {
            
            $result[] = new Combination(new CombinationId($row['products_properties_combis_id']),
                new CombinationOrder((int)$row['sort_order']),
                new CombinationModel($row['combi_model']),
                new CombinationEan($row['combi_ean']),
                new CombinationQuantity($row['combi_quantity']),
                $this->createSurcharge($row['combi_price_type'],(float)$row['combi_price'], $nonLinearSurcharge),
                $this->createVpe((int)$row['products_vpe_id'], (string)$row['products_vpe_name'], (float)$row['vpe_value'])
            );
        }


        return $result;
    }

    /**
     * @inheritDoc
     */
    public function hasProperties(ProductId $productId): bool
    {
        $result = $this->hasPropertiesCache($productId);
        if ($result === null) {
            $result = false;
            $records = $this->connection->createQueryBuilder()
                ->select('count(*) AS total')
                ->from('products_properties_combis', 'ppc')
                ->leftJoin('ppc',
                    'products_properties_combis_values',
                    'ppcv',
                    'ppc.products_properties_combis_id = ppcv.products_properties_combis_id')
                ->where('ppc.products_id = :products_id')
                ->setParameter('products_id', $productId->value())
                ->setMaxResults(1)
                ->execute()
                ->fetchAll();
            foreach ($records as $row) {
                $result = ((int)$row['total']) > 0;
            }

            $this->setHasPropertiesCache($productId, $result);
        }

        return $result;
    }

    /**
     * @param string $type
     * @param float $surcharge
     * @param bool $nonLinear
     * @return CombinationSurcharge|null
     */
    protected function createSurcharge(string $type, float $surcharge, bool $nonLinear): ?CombinationSurcharge
    {
        if (($surcharge || $nonLinear) && $type === 'calc') {
            return new CombinationSurcharge($surcharge, $nonLinear);
        }
        return null;
    }
    
    
    /**
     * @param int    $id
     * @param string $name
     * @param float  $value
     *
     * @return Vpe|null
     */
    protected function createVpe(int $id, string $name, float $value): ?Vpe
    {
        if ($id && $name && $value) {
            return new Vpe($id, $name, $value);
        }
        return null;
    }

    /**
     * @param int $combinationId
     *
     * @return mixed
     */
    protected function getCombinationModifierIdsCached(int $combinationId)
    {

        return $this->cache['getCombinationModifierIds'][$combinationId] ?? null;
    }

    /**
     * @param int $productId
     * @param bool|null $hasSurcharge
     * @return bool
     */
    protected function hasNonLinearSurcharge(int $productId, ?bool $hasSurcharge = null): bool
    {
        $result = $hasSurcharge ?? $this->hasSurcharge($productId);
        if ($result) {
            $sql = "SELECT
                    round((p.products_price + ppc.combi_price) / ppc.vpe_value,2) AS base_price
                FROM
                    products AS p
                LEFT JOIN
                    products_properties_combis AS ppc ON (ppc.products_id = p.products_id)
                WHERE
                    p.products_id = {$productId}
                    AND ppc.vpe_value != 0
                GROUP BY
                    base_price";
            try {
                $data = $this->connection->query($sql)->fetchAll();
                $countData = count($data);
                $result = $countData > 1 || $countData === 0;
            } catch (DBALException $e) {
                echo $e->getMessage();
            }
        }
        return $result;
    }

    /**
     * @param ProductId $productId
     *
     * @return mixed
     */
    protected function hasPropertiesCache(ProductId $productId)
    {
        return $this->cache['hasProperties'][$productId->value()] ?? null;
    }

    /**
     * @param int $productId
     * @return bool
     * @throws
     */
    protected function hasSurcharge(int $productId): bool
    {

        $sql = "SELECT
                    combi_price
                FROM
                    products_properties_combis
                WHERE
                    products_id = {$productId}
                GROUP BY
                    combi_price";
        try {
            return count($this->connection->query($sql)->fetchAll()) > 1;
        } catch (DBALException $e) {
            return true;
        }


    }

    /**
     * @param int $combinationId
     * @param     $result
     */
    protected function setCombinationModifierIdsCached(int $combinationId, $result): void
    {
        $this->cache['getCombinationModifierIds'][$combinationId] = $result;
    }

    /**
     * @param ProductId $productId
     * @param           $result
     *
     * @return void
     */
    protected function setHasPropertiesCache(ProductId $productId, $result)
    {
        $this->cache['hasProperties'][$productId->value()] = $result;
    }
    
    
    /**
     * @return int
     */
    protected function activeLanguageId(): int
    {
        return (int)$_SESSION['languages_id'];
    }
}